;;*****************************************************************************
;;*****************************************************************************
;;  FILENAME:   LCD.asm
;;  Version: 1.60, Updated on 2011/6/28 at 6:9:34
;;  Generated by PSoC Designer 5.1.2306
;;
;;  DESCRIPTION: LCD User Module software implementation file.
;;
;; This set of functions is written for the common 2 and 4 line
;; LCDs that use the Hitachi HD44780A controller.
;;
;;  LCD connections to PSoC port
;;
;;    PX.0 ==> LCD D4
;;    PX.1 ==> LCD D5
;;    PX.2 ==> LCD D6
;;    PX.3 ==> LCD D7
;;    PX.4 ==> LCD E
;;    PX.5 ==> LCD RS
;;    PX.6 ==> LCD R/W
;;
;;  NOTE: User Module APIs conform to the fastcall16 convention for marshalling
;;        arguments and observe the associated "Registers are volatile" policy.
;;        This means it is the caller's responsibility to preserve any values
;;        in the X and A registers that are still needed after the API functions
;;        returns. For Large Memory Model devices it is also the caller's 
;;        responsibility to preserve any value in the CUR_PP, IDX_PP, MVR_PP and 
;;        MVW_PP registers. Even though some of these registers may not be modified
;;        now, there is no guarantee that will remain the case in future releases.
;;-----------------------------------------------------------------------------
;;  Copyright (c) Cypress Semiconductor 2011. All Rights Reserved.
;;*****************************************************************************
;;*****************************************************************************

include "m8c.inc"
include "memory.inc"
include "LCD.inc"

;-----------------------------------------------
;  Global Symbols
;-----------------------------------------------

export   LCD_Start
export  _LCD_Start
export   LCD_Init
export  _LCD_Init

export   LCD_WriteData
export  _LCD_WriteData

export   LCD_Control
export  _LCD_Control

export  LCD_PrString
export _LCD_PrString

export  LCD_PrCString
export _LCD_PrCString

export  LCD_Position
export _LCD_Position

export  LCD_PrHexByte
export _LCD_PrHexByte

export  LCD_PrHexInt
export _LCD_PrHexInt

export  LCD_Delay50uTimes
export _LCD_Delay50uTimes

export  LCD_Delay50u
export _LCD_Delay50u

;-----------------------------------------------
; If bargraph functions not required, don't
; export the function names.
;-----------------------------------------------

IF (LCD_BARGRAPH_ENABLE)
export  LCD_InitBG
export _LCD_InitBG

export  LCD_InitVBG
export _LCD_InitVBG

; NOTE: The two functions,
;
;    LCD_DrawVBG and
;    LCD_DrawBG
;
; are implemented using both fastcall16 and legacy fastcall16 because they
; fall into a special and rare case where the calling sequences specified
; by the two disciplines are incompatible. The fastcall16 versions are
; provided for both C and Assembly users in all memory models. The legacy
; fastcall16 versions are provided only to support existing small memory
; model assembly language code---they do not work in the large memory
; model.
;
; ** The legacy fastcall16 versions are provided on a temporary basis to
; ** ease the transition to the 4.2 release of PSoC Designer. Their use is
; ** deprecated and their status is "No Further Maintenance".
;
; The fastcall16 versions of these functions are distinguished by a
; leading underscore in the name. The legacy fastcall16 names (which appear
; in this comment) do not have the leading underscore. Details on the
; calling sequence to be used for fastcall16 are given in the user module
; datasheet.
;
; Fastcall16 versions:
export _LCD_DrawVBG
export _LCD_DrawBG

IF SYSTEM_SMALL_MEMORY_MODEL
; Legacy Fastcall versions:
export  LCD_DrawVBG
export  LCD_DrawBG
ENDIF  ; SYSTEM_SMALL_MEMORY_MODEL

ENDIF  ; BARGRAPH_ENABLE

;
; The following functions are deprecated and will be eliminated in a future
; version of PSoC Designer.
;
export   LCD_Write_Data
export  _LCD_Write_Data


;-----------------------------------------------
;  EQUATES
;-----------------------------------------------

LCD_Port:           equ    PRT2DR
LCD_PortMode0:      equ    PRT2DM0
LCD_PortMode1:      equ    PRT2DM1

LCD_E:              equ    10h
LCD_RW:             equ    40h
LCD_RS:             equ    20h

LCD_DATA_MASK:      equ    0Fh
LCD_READY_BIT:      equ    08h

LCD_DATA_READ:      equ    ( LCD_E | LCD_RW | LCD_RS )
LCD_CNTL_READ:      equ    ( LCD_E | LCD_RW )
LCD_PORT_WRITE:     equ    7Fh
LCD_PORT_MASK:      equ    7Fh

LCD_DISP_INC:       equ    03h
LCD_DISP_OFF:       equ    08h
LCD_DISP_ON:        equ    0Ch
LCD_4BIT_2LINE:     equ    2Ch


;-----------------------------------------------
;      Bargraph definitions
;-----------------------------------------------

LCD_BG_CHAR_WIDTH:  equ    16     ; 16 characters in width
LCD_BG_SEG_WIDTH:   equ    80     ; 16 * 5 = 80
LCD_BG_COL_START:   equ     0     ; Always start in the left most column

                                  ; Offsets for 2x16, 2x20, 4x20
                                  ; Change these values for a custom LCD

LCD_ROW1_OFFSET:    equ    80h    ; Address/command offset for row 1
LCD_ROW2_OFFSET:    equ    C0h    ; Address/command offset for row 2
LCD_ROW3_OFFSET:    equ    94h    ; Address/command offset for row 1
LCD_ROW4_OFFSET:    equ    D4h    ; Address/command offset for row 2

LCD_BG_ROW1_OFFSET: equ    80h    ; Address/command offset for row 1
LCD_BG_ROW2_OFFSET: equ    C0h    ; Address/command offset for row 2

LCD_CG_RAM_OFFSET:  equ    40h    ; Offset to character RAM

AREA UserModules (ROM, REL)

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_PrCString
;
;  DESCRIPTION:
;    Print constant (ROM) string to LCD
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A:X  Pointer to String
;          A contains MSB of string address
;          X contains LSB of string address
;
;  RETURNS:  none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
 LCD_PrCString:
_LCD_PrCString:
    RAM_PROLOGUE RAM_USE_CLASS_1
 .Loop_PrCString:
    push  A                            ; Store ROM pointer
    push  X
    romx                               ; Get character from ROM
    jnz   .LCD_PrCString_WR            ; print character and advance pointer
    pop   X                            ; Restore the stack
    pop   A
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret                                ; Return

.LCD_PrCString_WR:
    call  LCD_WriteData                ; Write data to LCD
    pop   X                            ; Get ROM pointer
    pop   A
    inc   X                            ; Inc LSB of pointer
    jnc   .Loop_PrCString
    inc   A                            ; Inc MSB of pointer if LSB overflow
    jmp   .Loop_PrCString

.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_PrHexByte
;
;  DESCRIPTION:
;     Print a byte in Hex (two characters) to current LCD position
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A  => (BYTE) Data/char to be printed
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
.LITERAL
LCD_HEX_STR::
     DS    "0123456789ABCDEF"
.ENDLITERAL
.SECTION

 LCD_PrHexByte:
_LCD_PrHexByte:
    RAM_PROLOGUE RAM_USE_CLASS_1
    push  A                            ; Save lower nibble
    asr   A                            ; Shift high nibble to right
    asr   A
    asr   A
    asr   A
    and   A,0Fh                        ; Mask off nibble
    index LCD_HEX_STR                  ; Get Hex value
    call  LCD_WriteData                ; Write data to screen
    pop   A                            ; Restore value
    and   A,0Fh                        ; Mask off lower nibble
    index LCD_HEX_STR                  ; Get Hex value
    call  LCD_WriteData                ; Write data to screen
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_PrHexInt
;
;  DESCRIPTION:
;     Print an Int in Hex (four characters) to current LCD position
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A:X Integer value
;         A  contains LSB of Int
;         X  contains MSB of Int
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
 LCD_PrHexInt:
_LCD_PrHexInt:
    RAM_PROLOGUE RAM_USE_CLASS_1
    swap  A,X
    call  LCD_PrHexByte                ; Print MSB
    mov   A,X                          ; Move LSB into position
    call  LCD_PrHexByte                ; Print LSB
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_PrString
;
;  DESCRIPTION:
;     Print (RAM) ASCII string to LCD
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A:X contains pointer to string
;         X  contains LSB of string pointer
;         A  contains MSB or page of string pointer (not used at this time)
;
;  RETURNS:
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;          IDX_PP
;
;
 LCD_PrString:
_LCD_PrString:
    RAM_PROLOGUE RAM_USE_CLASS_3
    RAM_SETPAGE_IDX A
 .Loop_PrString:
    mov   A,[X]                        ; Get value pointed to by X
    jz    .End_LCD_PrString            ; Check for end of string
    ;LCD_writeData is known not to modify X so no need to preserve
    call  LCD_WriteData                ; Write data to screen
    inc   X                            ; Advance pointer to next character
    jmp   .Loop_PrString               ; Go get next character
.End_LCD_PrString:
    RAM_EPILOGUE RAM_USE_CLASS_3
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_WriteData
;
;  DESCRIPTION:
;     Write a byte to the LCD's data register.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;    A contains byte to be written to LCD data register
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
 LCD_WriteData:
_LCD_WriteData:
 LCD_Write_Data:   ; Do not use
_LCD_Write_Data:   ; Do not use
    RAM_PROLOGUE RAM_USE_CLASS_1
    call  LCD_Check_Ready              ; Make sure controller is ready
                                       ; A is preserved in LCD_Check_Ready
    push  A                            ; Save copy of character
    asr   A                            ; Shift high nibble to right
    asr   A
    asr   A
    asr   A
    and   A,0Fh                        ; Mask off high nibble
    call  LCD_WDATA_Nibble             ; Write Upper nibble
    pop   A                            ; Retrieve copy of character
    and   A,0Fh                        ; Mask off high nibble
    nop
    nop
    nop
    call  LCD_WDATA_Nibble   ; Write Lower nibble
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Control
;
;  DESCRIPTION:
;     Write a byte to the LCD's control register.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A contains data to be written to LCD control register.
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
 LCD_Control:
_LCD_Control:
    RAM_PROLOGUE RAM_USE_CLASS_1
    call  LCD_Check_Ready              ; Make sure controller is ready
                                       ; A is preserved in LCD_Check_Ready
    push  A                            ; Save copy of byte
    asr   A                            ; Shift Upper Nibble to right
    asr   A
    asr   A
    asr   A
    and   A,0Fh                        ; Mask off, just in case
    call  LCD_WCNTL_Nibble             ; Write high nibble
    pop   A                            ; Restore copy of byte
    and   A,0Fh                        ; Mask off high nibble
    nop
    nop
    nop
    call  LCD_WCNTL_Nibble             ; Write Lower nibble
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_WCNTL_Nibble
;
;  DESCRIPTION:
;     Write a single nibble to the LCD's command register
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A[3:0]   Contains Nibble to be written to command register
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
 LCD_WCNTL_Nibble:
    RAM_PROLOGUE RAM_USE_CLASS_4
    push  A
    RAM_SETPAGE_CUR >Port_2_Data_SHADE           ; Set CUR_PP to LCD variable address
    and   [Port_2_Data_SHADE],~LCD_PORT_MASK
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A                        ; Reset control lines

    pop   A
    and   A,LCD_DATA_MASK                        ; Make sure no bogus data in MSN
    or    A,LCD_E                                ; Bring "E" Enable line high
    or    A,[Port_2_Data_SHADE]                  ; OR in bit 7 just
    mov   reg[LCD_Port], A                       ; Write data
    mov   [Port_2_Data_SHADE],A                  ; Keep shadow register in sync
    nop
    and   A,(~LCD_PORT_MASK|LCD_DATA_MASK)       ; Disable E signal and leave data on bus.
    mov   [Port_2_Data_SHADE],A                  ; Keep shadow register in sync
    mov   reg[LCD_Port],A
    RAM_EPILOGUE RAM_USE_CLASS_4
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_WDATA_Nibble
;
;  DESCRIPTION:
;     Write a single nibble to the LCD's DATA register
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A[3:0]   Contains Nibble to be written to data register
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
LCD_WDATA_Nibble:
    RAM_PROLOGUE RAM_USE_CLASS_4
    push  A
    RAM_SETPAGE_CUR >Port_2_Data_SHADE          ; Set CUR_PP to LCD variable address
    and   [Port_2_Data_SHADE],~LCD_PORT_MASK
    or    [Port_2_Data_SHADE],LCD_RS                  ; Raise RS to signify a Data Write
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A

    pop   A
    and   A,LCD_DATA_MASK                             ; Make sure no bogus data in A[7:4]
    or    A,(LCD_E | LCD_RS)                          ; Bring "E" Enable line high
    or    A,[Port_2_Data_SHADE]                       ; Keep shadow in sync
    mov   reg[LCD_Port], A                            ; Write data
    mov   [Port_2_Data_SHADE],A                       ; Keep shadow in sync
    NOP
    and   A,(~LCD_PORT_MASK|LCD_DATA_MASK|LCD_RS)     ; Disable E signal and leave Data on bus
    mov   [Port_2_Data_SHADE],A                       ; keep shadow in sync
    mov   reg[LCD_Port],A
    RAM_EPILOGUE RAM_USE_CLASS_4
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Check_Ready
;
;  DESCRIPTION:
;     Wait until LCD has completed last command.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS: none
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
;     If LCD is not present, this routine may never return.
;
LCD_Check_Ready:
    RAM_PROLOGUE RAM_USE_CLASS_4
    push  A                                           ; Save Accumulator
    RAM_SETPAGE_CUR >Port_2_Data_SHADE                ; Set CUR_PP to LCD variable address
    and   [Port_2_Data_SHADE],~LCD_PORT_MASK          ; Mask of all LCD bits
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A                             ; Zero LCD port bits

    and   [Port_2_DriveMode_0_SHADE],~LCD_DATA_MASK   ; Clear out LCD mode bits.
    mov   A,[Port_2_DriveMode_0_SHADE]
    M8C_SetBank1                                      ; Change port mode to read status
    mov   reg[LCD_PortMode0],A                        ; Setup LCD Port for reading
    M8C_SetBank0

    or    [Port_2_Data_SHADE],LCD_RW                  ; Raise RW to signify Read operation
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A
    NOP

    push  X							                            ; Save 'X' register
    mov   X,255                         			            ; 255 Attempts

.LCD_RDY_LOOP:
    or    [Port_2_Data_SHADE], LCD_CNTL_READ                    ; Raise E to start cycle
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A

    nop                                               ; Wait 2 nops to make sure data is ready
    nop
    mov   A,reg[LCD_Port]

; The code below is used to work around the async read issue with the ICE with the 
; 25/26xxx family of devices.  It will help to eliminate "Invalid memory reference" 
; errors.  It is not required when running without the ICE or when using any other 
; family besides the 25/26xxx family. If not using the ICE or with any other family
; the ICE_PORT_SYNC flag should be set to 0.
IF(ICE_PORT_SYNC)                          
    mov   reg[ 0xfa], A                    
    mov   A, reg[0xfa]                     
ENDIF   
                                   
    push  A
    and   [Port_2_Data_SHADE],(~LCD_PORT_MASK | LCD_RW)         ; Lower E signal
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A

    nop                                    ; Add delay for the slowest part and the
    nop                                    ; fastest PSoC
    nop
                                                                ; Get the LSBs
    or    [Port_2_Data_SHADE],LCD_CNTL_READ                     ; Raise E to start cycle
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A

    nop
    nop

    and   [Port_2_Data_SHADE],(~LCD_PORT_MASK | LCD_RW)         ; Lower E signal
    mov   A,[Port_2_Data_SHADE]
    mov   reg[LCD_Port],A

    pop   A
    and   A,LCD_READY_BIT                                       ; Check busy

    jz    .UNLOCK
    dec   X
    jnz   .LCD_RDY_LOOP                                         ; If LCD still busy, read again for 255 times
.UNLOCK:
    pop   X							                            ; Restore 'X' register

    or    [Port_2_DriveMode_0_SHADE],LCD_PORT_WRITE   ; Revert Data bit to Write mode
    mov   A,[Port_2_DriveMode_0_SHADE]
    M8C_SetBank1
    mov   reg[LCD_PortMode0],A                        ; Setup LCD Port for writing
    M8C_SetBank0
    pop   A
    RAM_EPILOGUE RAM_USE_CLASS_4                                ; Restore Accumulator
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Start
;  FUNCTION NAME: LCD_Init
;
;  DESCRIPTION:
;     Initialize LCD
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS: none
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
;  THEORY of OPERATION or PROCEDURE:
;    REGISTERS ARE VOLATILE: THE A AND X REGISTERS MAY BE MODIFIED!
;    This initialization is a bit long, but it should work for
;    most 2 and 4 line LCDs.
;
 LCD_Start:
_LCD_Start:
 LCD_Init:
_LCD_Init:
    RAM_PROLOGUE RAM_USE_CLASS_4
    RAM_SETPAGE_CUR >Port_2_Data_SHADE                          ; Set CUR_PP to LCD variable address

    and   [Port_2_DriveMode_0_SHADE],~LCD_PORT_MASK             ; Mask off LCD bits
    or    [Port_2_DriveMode_0_SHADE],LCD_PORT_WRITE             ; Set LCD port for writing
    and   [Port_2_DriveMode_1_SHADE],~LCD_PORT_MASK             ; Mask off LCD bits

    mov   A,[Port_2_DriveMode_0_SHADE]
    M8C_SetBank1
    mov   reg[LCD_PortMode0],A                                  ; Setup LCD Port for writing
    mov   A,[Port_2_DriveMode_1_SHADE]
    mov   reg[LCD_PortMode1],A
    M8C_SetBank0

    mov   A,250                             ; Delay for 12.5 mSec (250 * 50uSec)
    call  LCD_Delay50uTimes
    mov   A,250                             ; Delay for 12.5 mSec (250 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,03h
    call  LCD_WCNTL_Nibble

    mov   A,82                              ; Delay for 4.1 mSec (82 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,03h
    call  LCD_WCNTL_Nibble

    call  LCD_Delay50u
    call  LCD_Delay50u
    call  LCD_Delay50u

    mov   A,03h
    call  LCD_WCNTL_Nibble

    mov   A,90                              ; Delay for 4.5 mSec (90 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,02h
    call  LCD_WCNTL_Nibble

    mov   A,90                              ; Delay for 4.5 mSec (90 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,08h
    call  LCD_Control
    mov   A,90                              ; Delay for 4.5 mSec (90 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,01h
    call  LCD_Control
    mov   A,90                              ; Delay for 4.5 mSec (90 * 50uSec)
    call  LCD_Delay50uTimes

    mov   A,06h
    call  LCD_Control

    mov   A,0Eh
    call  LCD_Control

    mov   A,LCD_4BIT_2LINE                  ; Setup for 4 bit interface, 2 line
    call  LCD_Control

    mov   A,LCD_DISP_OFF
    call  LCD_Control

    mov   A,LCD_DISP_ON
    call  LCD_Control

    mov   A,LCD_DISP_INC
    call  LCD_Control

    mov   A,90                              ; Delay for 4.5 mSec (90 * 50uSec)
    call  LCD_Delay50uTimes
    RAM_EPILOGUE RAM_USE_CLASS_4
    ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Position
;
;  DESCRIPTION:
;     Position Cursor at Row and Col location
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A => Row  0 to 3
;     X => Col  0 to 39+
;
;  RETURNS:  none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;
.LITERAL
LCD_ROW_OFFSET::
     DB    LCD_ROW1_OFFSET, LCD_ROW2_OFFSET, LCD_ROW3_OFFSET, LCD_ROW4_OFFSET
.ENDLITERAL

.SECTION
 LCD_Position:
_LCD_Position:
    RAM_PROLOGUE RAM_USE_CLASS_2
    and   A,03h                        ; Mask off 2 bits for row address 0 to 3
    push  X                            ; Store COL
    index LCD_ROW_OFFSET ; Get ROW memory offset from table
    mov   X,SP                         ; Get Stack pointer
    add   A,[X+(-1)]                   ; Add the COL to the display pointer
    pop   X

    call  LCD_Control                  ; Write control byte
    RAM_EPILOGUE RAM_USE_CLASS_2
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Delay50uTimes
;
;  DESCRIPTION:
;     Delay increments of 50uSeconds
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A contains the delay multiplier
;
;  RETURNS:
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;
;
 LCD_Delay50uTimes:
_LCD_Delay50uTimes:
    RAM_PROLOGUE RAM_USE_CLASS_1
    call  LCD_Delay50u
    dec   A
    jnz   LCD_Delay50uTimes
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret

.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_Delay50u
;
;  DESCRIPTION:
;     Delay 50uSec for any clock frequency from 1.5MHz to 24MHz
;     Slower clock frequencies the delay will be;
;           1.5
;        -------------- * 50uSec
;        clock_freq(MHz)
;
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS: none
;
;  RETURNS: none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;
;  THEORY of OPERATION or PROCEDURE:
;
.LITERAL
 LCD_Delay50u_Table::
     DB    08h,  19h,   3Ah,   7Ch,   01h,    01h,    01h,   01h
;         3MHz, 6MHz, 12MHz, 24MHz, 1.5MHz, 750kHz, 188kHz, 94kHz
.ENDLITERAL
.SECTION

  LCD_Delay50u:
 _LCD_Delay50u:                        ; [11]  Call
    RAM_PROLOGUE RAM_USE_CLASS_1
    push  A
    M8C_SetBank1                         ; [4]
    mov   A, reg[OSC_CR0]                ; [6] Get delay value
    M8C_SetBank0                         ; [4]
    and   A,07h                          ; [4] Mask off only the clock bits
    cmp   A,05h
    jnc   .Delay50u_End
    index LCD_Delay50u_Table ; [13] Get delay value
.Delay50u_Loop:                          ;
    dec   A                              ; [4]
    jnz   .Delay50u_Loop                 ; [5]
.Delay50u_End:
    pop   A
    RAM_EPILOGUE RAM_USE_CLASS_1
    ret
.ENDSECTION


;-----------------------------------------------------------------------------
;      If bargraph is not enabled, the following functions are not required.
;-----------------------------------------------------------------------------

IF (LCD_BARGRAPH_ENABLE)

IF SYSTEM_SMALL_MEMORY_MODEL
.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_DrawBG
;
;  DESCRIPTION:
;  This legacy fastcall version are provided only to support existing small
;  memory model assembly language code---it does not work in the large memory
;  model.
;
;  ** This legacy fastcall version is provided on a temporary basis to
;  ** ease the transition to the 4.2 release of PSoC Designer. Its use is
;  ** deprecated and its status is "No Further Maintenance". If you call this
;  ** function in assembly you should convert to _LCD_DrawVBG
;  ** (with a leading underscore) and the fastcall16 interface
;
;  Draw a horizontal bargraph on the LCD with the given parameters.  This
;  is a legacy function that is intended to support existing Assembly
;  language programs that call this function.  This should not be used for
;  new code or with Large Memory Model programs.
;-----------------------------------------------------------------------------
;
;  LEGACY FASTCALL ARGUMENTS:
;    A    => Starting row for bargraph 0 to 3
;   [X]   => Starting Column for bargraph 0 to 39+
;   [x-1] => Length of bargraph in chars 1 to 40+
;   [X-2] => Position of pointer in segments 5 times Length
;
;
;  RETURNS:  none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;
;    If LCD_Init is not called before this function, the
;    bargraph will not be drawn properly.
;
; Stack offset constants
BG_COLX:       equ  0                   ; Stack position of Column
BG_CHAR_LENX:  equ -1                   ; Stack position of Length
BG_LENGTHX:    equ -2                   ; Stack position of bargraph pointer position


LCD_DrawBG:
    push  X
    mov   X,[X+BG_COLX]                 ; Row in A, Col in X
    call  LCD_Position                 ; Set cursor position
    pop   X                             ; Restore pointer

.LCD_BG_LOOP1X:
    cmp   [X+BG_LENGTHX],00h            ; Check for past end of BG
    jnz   .LCD_CHECK1X
    mov   A,00h                         ; Load empty character
    jmp   .LCD_BG_DOITX                 ;

.LCD_CHECK1X:
    cmp   [X+BG_LENGTHX],06h            ; Check if BG pointer is at this character
    jnc   .LCD_CHECK2X                  ; Note yet, use full character
    mov   A,[X+BG_LENGTHX]
    sub   [X+BG_LENGTHX],A
    jmp   .LCD_BG_DOITX

.LCD_CHECK2X:                           ; Put index to full character
    mov   A, 06h
    sub   [X+BG_LENGTHX],05h            ; Subtract another 5 positions

.LCD_BG_DOITX:
    call  LCD_WriteData                ; Display BG character

    dec   [X+BG_CHAR_LENX]              ; Dec Char count
    jnz   .LCD_BG_LOOP1X                ; Do it all over again
    ret
.ENDSECTION
ENDIF ; SYSTEM_SMALL_MEMORY_MODEL

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_DrawBG
;
;  DESCRIPTION:
;     Draw a horizontal bargraph on the LCD with the given parameters.
;
;
;-----------------------------------------------------------------------------
;
;  FASTCALL16 ARGUMENTS:
;   [SP-3] => Starting row for bargraph 0 to 3
;   [SP-4] => Starting Column for bargraph 0 to 39+
;   [SP-5] => Length of bargraph in chars 1 to 40+
;   [SP-6] => Position of pointer in segments 5 times Length
;
;
;  RETURNS:  none
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
;    If LCD_Init is not called before this function, the
;    bargraph will not be drawn properly.
;
; Stack offset constants
BG_ROW:       equ -3
BG_COL:       equ -4                   ; Stack position of Column
BG_CHAR_LEN:  equ -5                   ; Stack position of Length
BG_LENGTH:    equ -6                   ; Stack position of bargraph pointer position


_LCD_DrawBG:
    RAM_PROLOGUE RAM_USE_CLASS_2
    mov   X, SP
    push  X
    mov   A,[X+BG_ROW]                 ; Row in A
    mov   X,[X+BG_COL]                 ; Col in X
    RAM_EPILOGUE RAM_USE_CLASS_2
    call  LCD_Position                 ; Set cursor position
    RAM_PROLOGUE RAM_USE_CLASS_2
    pop  X

.LCD_BG_LOOP1:
    cmp   [X+BG_LENGTH],00h            ; Check for past end of BG
    jnz   .LCD_CHECK1
    mov   A,00h                        ; Load empty character
    jmp   .LCD_BG_DOIT                  ;

.LCD_CHECK1:
    cmp   [X+BG_LENGTH],06h            ; Check if BG pointer is at this character
    jnc   .LCD_CHECK2                   ; Note yet, use full character
    mov   A,[X+BG_LENGTH]
    sub   [X+BG_LENGTH],A
    jmp   .LCD_BG_DOIT

.LCD_CHECK2:                            ; Put index to full character
    mov   A, 06h
    sub   [X+BG_LENGTH],05h            ; Subtract another 5 positions

.LCD_BG_DOIT:
    call  LCD_WriteData                ; Display BG character

    dec   [X+BG_CHAR_LEN]              ; Dec Char count
    jnz   .LCD_BG_LOOP1                 ; Do it all over again
    RAM_EPILOGUE RAM_USE_CLASS_2
    ret
.ENDSECTION

IF SYSTEM_SMALL_MEMORY_MODEL
.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_DrawVBG
;
;  DESCRIPTION:
;  This legacy fastcall version are provided only to support existing small
;  memory model assembly language code---it does not work in the large memory
;  model.
;
;  ** This legacy fastcall version is provided on a temporary basis to
;  ** ease the transition to the 4.2 release of PSoC Designer. Its use is
;  ** deprecated and its status is "No Further Maintenance". If you call this
;  ** function in assembly you should convert to _LCD_DrawVBG
;  ** (with a leading underscore) and the fastcall16 interface
;
;  Draw a vertical bargraph on the LCD with the given parameters. This
;  is a legacy function that is intended to support existing Assembly
;  language programs that call this function.  This should not be used for
;  new code or with Large Memory Model programs.
;-----------------------------------------------------------------------------
;
;  LEGACY FASTCALL ARGUMENTS:
;    A    => Starting row for bargraph 0 to 3
;   [X]   => Starting Column for bargraph 0 to 40+
;   [x-1] => Height of bargraph in chars 1 - 4
;   [X-2] => Position of pointer in segments 8 times height
;  RETURNS:
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;    
;    If LCD_Init is not called before this function, the
;    bargraph will not be drawn properly.
;
; Stack offset constants
VBG_COLX:            equ  0
VBG_CHAR_HEIGHTX:    equ -1
VBG_SEG_HEIGHTX:     equ -2

LCD_DrawVBG:

    and   A,03h                        ; Make sure only rows 0 - 3 are valid
.VBG_LOOPX:
    push  A
    index LCD_ROW_OFFSET  ; Get row offset
    add   A,[X+VBG_COLX]                ; Add column offset to position
    call  LCD_Control                  ; Position Cursor
    cmp   [X+VBG_SEG_HEIGHTX],00h       ; Check for zero segs
    jnz   .VBG_NZ_SEGX
    mov   A,' '                        ; Load space character
    jmp   .VBG_WRITE_CHARX
.VBG_NZ_SEGX:
    cmp   [X+VBG_SEG_HEIGHTX],09h       ; Check for full segment
    jnc   .VBG_FULL_SEGX
                                        ; Partial segment between 1 and 8
    mov   A,[X+VBG_SEG_HEIGHTX]
    dec   A
    mov   [X+VBG_SEG_HEIGHTX],00h       ; Zero segment height
    jmp   .VBG_WRITE_CHARX

.VBG_FULL_SEGX:                          ; Bargaph
    sub   [X+VBG_SEG_HEIGHTX],08h       ; Subtract full segment
    mov   A,07h                        ; Load full segment

.VBG_WRITE_CHARX:                        ; Write character to display
    call  LCD_WriteData                ; Write value
    pop   A
    dec   A
    dec   [X+VBG_CHAR_HEIGHTX]
    jnz   .VBG_LOOPX
    ret
.ENDSECTION
ENDIF ; SYSTEM_SMALL_MEMORY_MODEL

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_DrawVBG
;
;  DESCRIPTION:
;     Draw a vertical bargraph on the LCD with the given parameters.
;
;
;-----------------------------------------------------------------------------
;
;  FASTCALL16 ARGUMENTS:
;
;   [SP-3] => Starting row for bargraph 0 to 3
;   [SP-4] => Starting Column for bargraph 0 to 40+
;   [SP-5] => Height of bargraph in chars 1 - 4
;   [SP-6] => Position of pointer in segments 8 times height
;  RETURNS:
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
;    If LCD_Init is not called before this function, the
;    bargraph will not be drawn properly.
;
; Stack offset constants
VBG_ROW:        equ -3
VBG_COL:            equ -4
VBG_CHAR_HEIGHT:    equ -5
VBG_SEG_HEIGHT:     equ -6

_LCD_DrawVBG:
    RAM_PROLOGUE RAM_USE_CLASS_2
    mov   X, SP
    mov   A, [X+VBG_ROW]
    and   A,03h                        ; Make sure only rows 0 - 3 are valid
.VBG_LOOP:
    push  A
    index LCD_ROW_OFFSET  ; Get row offset
    add   A,[X+VBG_COL]                ; Add column offset to position
    call  LCD_Control                  ; Position Cursor
    cmp   [X+VBG_SEG_HEIGHT],00h       ; Check for zero segs
    jnz   .VBG_NZ_SEG
    mov   A,' '                        ; Load space character
    jmp   .VBG_WRITE_CHAR
.VBG_NZ_SEG:
    cmp   [X+VBG_SEG_HEIGHT],09h       ; Check for full segment
    jnc   .VBG_FULL_SEG
                                       ; Partial segment between 1 and 8
    mov   A,[X+VBG_SEG_HEIGHT]
    dec   A
    mov   [X+VBG_SEG_HEIGHT],00h       ; Zero segment height
    jmp   .VBG_WRITE_CHAR

.VBG_FULL_SEG:                          ; Bargaph
    sub   [X+VBG_SEG_HEIGHT],08h       ; Subtract full segment
    mov   A,07h                        ; Load full segment

.VBG_WRITE_CHAR:                        ; Write character to display
    call  LCD_WriteData                ; Write value
    pop   A
    dec   A
    dec   [X+VBG_CHAR_HEIGHT]
    jnz   .VBG_LOOP
    RAM_EPILOGUE RAM_USE_CLASS_2
    ret
.ENDSECTION

.SECTION
;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_InitVBG
;
;  DESCRIPTION:
;     Initialize the vertical bargraph characters.
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:  none
;
;  RETURNS:  none
;
;  SIDE EFFECTS:
;    REGISTERS ARE VOLATILE: THE A AND X REGISTERS MAY BE MODIFIED!
;    Only one type of bargraph (horizontal or vertical) may be used
;    at a time since they each require their own set of characters.
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
; Stack offset constants
VBGDATA_CTR:      equ    00h           ; Char data count stack offset
VBG_BYTES:        equ    01h           ; Byte counter stack offset

 LCD_InitVBG:
_LCD_InitVBG:
    RAM_PROLOGUE RAM_USE_CLASS_2
    mov   X,SP                         ; Get location of stack
    push  A                            ; Create 2 locations
    push  A

    mov   A,LCD_CG_RAM_OFFSET              ; Setup pointer
    call  LCD_Control                  ; Position the CG pointer
    mov   [X+VBGDATA_CTR],01h          ; Reset data counter

 .VBG_Loop1:                            ; loop once for each 8 characters
    mov   [X+VBG_BYTES],08h            ; Load cycle pointer
 .VBG_Loop2:                            ; Loop once for each line in character (8 times)
    mov   A,[X+VBGDATA_CTR]
    cmp   A,[X+VBG_BYTES]
    jnc   .VBG_SOLID
    mov   A,00h                        ; Empty line
    jmp   .VBG_Load                     ; Jump to load the bargraph
.VBG_SOLID:
    mov   A,FFh                        ; Load solid line
.VBG_Load:
    call  LCD_WriteData                ; character data
    dec   [X+VBG_BYTES]                ; Dec byte counter
    jnz   .VBG_Loop2                    ; End Loop 2
    inc   [X+VBGDATA_CTR]
    cmp   [X+VBGDATA_CTR],09h
    jnz   .VBG_Loop1                    ; End Loop1

    pop  A
    pop  A
    mov  A,LCD_DISP_ON                    ; Turn on display, don't really
    call LCD_Control                   ; need this.
    RAM_EPILOGUE RAM_USE_CLASS_2
    ret
.ENDSECTION

;-----------------------------------------------------------------------------
;  FUNCTION NAME: LCD_InitBG
;
;  DESCRIPTION:
;     Initialize horizontal bargraph characters
;
;-----------------------------------------------------------------------------
;
;  ARGUMENTS:
;     A = type  0 = full                   |||||||||........
;               1 = single vertical line   ..........|......
;
;  RETURNS:
;
;  SIDE EFFECTS:
;    The A and X registers may be modified by this or future implementations
;    of this function.  The same is true for all RAM page pointer registers in
;    the Large Memory Model.  When necessary, it is the calling function's
;    responsibility to preserve their values across calls to fastcall16 
;    functions.
;          
;    Currently only the page pointer registers listed below are modified: 
;          CUR_PP
;
;    Only one type of bargraph (horizontal or vertical) may be used
;    at a time since they each require their own set of characters.
;
;  THEORY of OPERATION or PROCEDURE:
;    This function writes to the LCD character RAM to generate 8 custom
;    characters used to generated one of two horizontal bargraphs.
;
.LITERAL
 LCD_BG_TYPE1:: ; ....., |...., ||..., |||.., ||||., |||||, |||||
     DB    00h, 10h, 18h, 1Ch, 1Eh, 1Fh, 1Fh
 LCD_BG_TYPE2:: ; ....., |...., .|..., ..|.., ...|., ....|, .....
    DB  00h, 10h, 08h, 04h, 02h, 01h, 00h
.ENDLITERAL

.SECTION
; Stack offset constants
BGDATA_PTR:   equ    00h               ; Stack offsets
BGCHARS:      equ    01h
BGTYPE:       equ    02h

 LCD_InitBG:
_LCD_InitBG:
    RAM_PROLOGUE RAM_USE_CLASS_2
    mov   X,SP                         ; Get location of stack
    add   SP,3
    mov   [X+BGTYPE],A                 ; Store the bargraph type

    mov   A,LCD_CG_RAM_OFFSET              ; Setup pointer
    call  LCD_Control                  ; Position the CG pointer
    mov   [X+BGDATA_PTR],00h           ; Reset pointer to BG data

 .BG_Loop1:
    mov   [X+BGCHARS],08h              ; Load cycle pointer
 .BG_Loop2:
    mov   A,[X+BGDATA_PTR]
    cmp   [X+BGTYPE],00h               ; Check which bargraph
    jnz   .BG_OTHER
    index LCD_BG_TYPE1
    jmp   .BG_Load
 .BG_OTHER:
    index LCD_BG_TYPE2
 .BG_Load:
    call  LCD_WriteData
    dec   [X+BGCHARS]                  ; Character builder counter
    jnz   .BG_Loop2
    inc   [X+BGDATA_PTR]               ; Advance to next character
    cmp   [X+BGDATA_PTR],07h
    jnz   .BG_Loop1

    add   SP,-3
    mov   A,LCD_DISP_ON
    call  LCD_Control
    RAM_EPILOGUE RAM_USE_CLASS_2
    ret
.ENDSECTION

ENDIF

; End of File LCD.asm
